/*
 * Firebase iOS Client Library
 *
 * Copyright © 2013 Firebase - All Rights Reserved
 * https://www.firebase.com
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binaryform must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY FIREBASE AS IS AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL FIREBASE BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */


#import <Foundation/Foundation.h>
#import "FQuery.h"
#import "FirebaseApp.h"
#import "FDataSnapshot.h"
#import "FMutableData.h"
#import "FTransactionResult.h"
#import "FAuthData.h"
#import "FAuthType.h"
#import "FirebaseServerValue.h"
#import "FConfig.h"

/**
 * A Firebase reference represents a particular location in your Firebase
 * and can be used for reading or writing data to that Firebase location.
 *
 * This class is the starting point for all Firebase operations. After you've
 * initialized it with initWithUrl: you can use it
 * to read data (ie. observeEventType:withBlock:), write data (ie. setValue:), and to create new
 * Firebase references (ie. child:).
 */
@interface Firebase : FQuery


/** @name Initializing a Firebase object */

/**
 * Initialize this Firebase reference with an absolute URL. 
 *
 * @param url The Firebase URL (ie: https://SampleChat.firebaseIO-demo.com)
 */
- (id)initWithUrl:(NSString *)url;

/** @name Getting references to children locations */

/**
 * Get a Firebase reference for the location at the specified relative path.
 * The relative path can either be a simple child key (e.g. 'fred') or a
 * deeper slash-separated path (e.g. 'fred/name/first').
 *
 * @param pathString A relative path from this location to the desired child location.
 * @return A Firebase reference for the specified relative path.
 */
- (Firebase *) childByAppendingPath:(NSString *)pathString;


/**
 * childByAutoId generates a new child location using a unique key and returns a
 * Firebase reference to it. This is useful when the children of a Firebase
 * location represent a list of items.
 *
 * The unique key generated by childByAutoId: is prefixed with a client-generated
 * timestamp so that the resulting list will be chronologically-sorted.
 *
 * @return A Firebase reference for the generated location.
 */
- (Firebase *) childByAutoId;


/** @name Writing data */

/*!  Write data to this Firebase location.

This will overwrite any data at this location and all child locations. 
 
Data types that can be set are:

- NSString -- @"Hello World"
- NSNumber (also includes boolean) -- @YES, @43, @4.333
- NSDictionary -- @{@"key": @"value", @"nested": @{@"another": @"value"} }
- NSArray

The effect of the write will be visible immediately and the corresponding
events will be triggered. Synchronization of the data to the Firebase 
servers will also be started.
 
Passing null for the new value is equivalent to calling remove:;
all data at this location or any child location will be deleted.

Note that setValue: will remove any priority stored at this location, so if priority
is meant to be preserved, you should use setValue:andPriority: instead.


**Server Values** - Placeholder values you may write into Firebase as a value or priority
that will automatically be populated by the Firebase Server.

- kFirebaseServerValueTimestamp - The number of milliseconds since the Unix epoch

 
@param value The value to be written.
 */
- (void) setValue:(id)value;


#define kFirebaseServerValueTimestamp @{ @".sv": @"timestamp" }

/**
 * The same as setValue: with a block that gets triggered after the write operation has
 * been committed to the Firebase servers.
 *
 * @param value The value to be written.
 * @param block The block to be called after the write has been committed to the Firebase servers.
 */
- (void) setValue:(id)value withCompletionBlock:(void (^)(NSError* error, Firebase* ref))block;


/**
 * The same as setValue: with an additional priority to be attached to the data being written.
 * Priorities are used to order items.
 *
 * @param value The value to be written.
 * @param priority The priority to be attached to that data.
 */
- (void) setValue:(id)value andPriority:(id)priority;


/**
 * The same as setValue:andPriority: with a block that gets triggered after the write operation has
 * been committed to the Firebase servers.
 *
 * @param value The value to be written.
 * @param priority The priority to be attached to that data.
 * @param block The block to be called after the write has been committed to the Firebase servers.
 */
- (void) setValue:(id)value andPriority:(id)priority withCompletionBlock:(void (^)(NSError* error, Firebase* ref))block;


/**
 * Remove the data at this Firebase location. Any data at child locations will also be deleted.
 * 
 * The effect of the delete will be visible immediately and the corresponding events
 * will be triggered. Synchronization of the delete to the Firebase servers will 
 * also be started.
 *
 * remove: is equivalent to calling setValue:nil
 */
- (void) removeValue;


/**
 * The same as remove: with a block that gets triggered after the remove operation has
 * been committed to the Firebase servers.
 *
 * @param block The block to be called after the remove has been committed to the Firebase servers.
 */
- (void) removeValueWithCompletionBlock:(void (^)(NSError* error, Firebase* ref))block;

/**
 * Set a priority for the data at this Firebase location.
 * Priorities can be used to provide a custom ordering for the children at a location
 * (if no priorities are specified, the children are ordered by key).
 *
 * You cannot set a priority on an empty location. For this reason
 * setValue:andPriority: should be used when setting initial data with a specific priority
 * and setPriority: should be used when updating the priority of existing data.
 *
 * Children are sorted based on this priority using the following rules:
 *
 * Children with no priority come first.
 * Children with a number as their priority come next. They are sorted numerically by priority (small to large).
 * Children with a string as their priority come last. They are sorted lexicographically by priority.
 * Whenever two children have the same priority (including no priority), they are sorted by key. Numeric
 * keys come first (sorted numerically), followed by the remaining keys (sorted lexicographically).
 * 
 * Note that priorities are parsed and ordered as IEEE 754 double-precision floating-point numbers.
 * Keys are always stored as strings and are treated as numbers only when they can be parsed as a
 * 32-bit integer
 *
 * @param priority The priority to set at the specified location.
 */
- (void) setPriority:(id)priority;


/**
 * The same as setPriority: with a block block that is called once the priority has
 * been committed to the Firebase servers.
 *
 * @param priority The priority to set at the specified location.
 * @param block The block that is triggered after the priority has been written on the servers.
 */
- (void) setPriority:(id)priority withCompletionBlock:(void (^)(NSError* error, Firebase* ref))block;

/**
 * Update changes the values of the keys specified in the dictionary without overwriting other
 * keys at this location.
 *
 * @param values A dictionary of the keys to change and their new values
 */
- (void) updateChildValues:(NSDictionary *)values;

/**
 * The same as update: with a block block that is called once the update has been committed to the 
 * Firebase servers
 *
 * @param values A dictionary of the keys to change and their new values
 * @param block The block that is triggered after the update has been written on the Firebase servers
 */
- (void) updateChildValues:(NSDictionary *)values withCompletionBlock:(void (^)(NSError* error, Firebase* ref))block;


/** @name Attaching observers to read data */

/*! observeEventType:withBlock: is used to listen for data changes at a particular location.
 
This is the primary way to read data from Firebase. Your block will be triggered
for the initial data and again whenever the data changes.
 
Use removeObserverWithHandle: to stop receiving updates.
 
Supported events types for all realtime observers are specified in FEventType as:

    typedef NS_ENUM(NSInteger, FEventType) {
      FEventTypeChildAdded,    // 0, fired when a new child node is added to a location
      FEventTypeChildRemoved,  // 1, fired when a child node is removed from a location
      FEventTypeChildChanged,  // 2, fired when a child node at a location changes
      FEventTypeChildMoved,    // 3, fired when a child node moves relative to the other child nodes at a location
      FEventTypeValue          // 4, fired when any data changes at a location and, recursively, any children
    };

@param eventType The type of event to listen for.
@param block The block that should be called with initial data and updates as a FDataSnapshot.
@return A handle used to unregister this block later using removeObserverWithHandle:
*/
- (FirebaseHandle) observeEventType:(FEventType)eventType withBlock:(void (^)(FDataSnapshot* snapshot))block;


/**
 * observeEventType:andPreviousSiblingKeyWithBlock: is used to listen for data changes at a particular location.
 * This is the primary way to read data from Firebase. Your block will be triggered
 * for the initial data and again whenever the data changes. In addition, for FEventTypeChildAdded, FEventTypeChildMoved, and
 * FEventTypeChildChanged events, your block will be passed the key of the previous node by priority order.
 *
 * Use removeObserverWithHandle: to stop receiving updates.
 *
 * @param eventType The type of event to listen for.
 * @param block The block that should be called with initial data and updates as a FDataSnapshot, as well as the
 * previous child's key.
 * @return A handle used to unregister this block later using removeObserverWithHandle:
 */
- (FirebaseHandle) observeEventType:(FEventType)eventType andPreviousSiblingKeyWithBlock:(void (^)(FDataSnapshot* snapshot, NSString* prevKey))block;


/**
 * observeEventType:withBlock: is used to listen for data changes at a particular location.
 * This is the primary way to read data from Firebase. Your block will be triggered
 * for the initial data and again whenever the data changes.
 *
 * The cancelBlock will be called if you will no longer receive new events due to no longer having permission.
 *
 * Use removeObserverWithHandle: to stop receiving updates.
 *
 * @param eventType The type of event to listen for.
 * @param block The block that should be called with initial data and updates as a FDataSnapshot.
 * @param cancelBlock The block that should be called if this client no longer has permission to receive these events
 * @return A handle used to unregister this block later using removeObserverWithHandle:
 */
- (FirebaseHandle) observeEventType:(FEventType)eventType withBlock:(void (^)(FDataSnapshot* snapshot))block withCancelBlock:(void (^)(NSError* error))cancelBlock;


/**
 * observeEventType:andPreviousSiblingKeyWithBlock: is used to listen for data changes at a particular location.
 * This is the primary way to read data from Firebase. Your block will be triggered
 * for the initial data and again whenever the data changes. In addition, for FEventTypeChildAdded, FEventTypeChildMoved, and
 * FEventTypeChildChanged events, your block will be passed the key of the previous node by priority order.
 *
 * The cancelBlock will be called if you will no longer receive new events due to no longer having permission.
 *
 * Use removeObserverWithHandle: to stop receiving updates.
 *
 * @param eventType The type of event to listen for.
 * @param block The block that should be called with initial data and updates as a FDataSnapshot, as well as the previous child's key.
 * @param cancelBlock The block that should be called if this client no longer has permission to receive these events
 * @return A handle used to unregister this block later using removeObserverWithHandle:
 */
- (FirebaseHandle) observeEventType:(FEventType)eventType andPreviousSiblingKeyWithBlock:(void (^)(FDataSnapshot* snapshot, NSString* prevKey))block withCancelBlock:(void (^)(NSError* error))cancelBlock;


/**
 * This is equivalent to observeEventType:withBlock:, except the block is immediately canceled after the initial data is returned.
 *
 * @param eventType The type of event to listen for.
 * @param block The block that should be called with initial data and updates as a FDataSnapshot.
 */
- (void) observeSingleEventOfType:(FEventType)eventType withBlock:(void (^)(FDataSnapshot* snapshot))block;


/**
 * This is equivalent to observeEventType:withBlock:, except the block is immediately canceled after the initial data is returned. In addition, for FEventTypeChildAdded, FEventTypeChildMoved, and
 * FEventTypeChildChanged events, your block will be passed the key of the previous node by priority order.
 *
 * @param eventType The type of event to listen for.
 * @param block The block that should be called with initial data and updates as a FDataSnapshot, as well as the previous child's key.
 */
- (void) observeSingleEventOfType:(FEventType)eventType andPreviousSiblingKeyWithBlock:(void (^)(FDataSnapshot* snapshot, NSString* prevKey))block;


/**
 * This is equivalent to observeEventType:withBlock:, except the block is immediately canceled after the initial data is returned.
 *
 * The cancelBlock will be called if you do not have permission to read data at this location.
 *
 * @param eventType The type of event to listen for.
 * @param block The block that should be called with initial data and updates as a FDataSnapshot.
 * @param cancelBlock The block that will be called if you don't have permission to access this data
 */
- (void) observeSingleEventOfType:(FEventType)eventType withBlock:(void (^)(FDataSnapshot* snapshot))block withCancelBlock:(void (^)(NSError* error))cancelBlock;


/**
 * This is equivalent to observeEventType:withBlock:, except the block is immediately canceled after the initial data is returned. In addition, for FEventTypeChildAdded, FEventTypeChildMoved, and
 * FEventTypeChildChanged events, your block will be passed the key of the previous node by priority order.
 *
 * The cancelBlock will be called if you do not have permission to read data at this location.
 *
 * @param eventType The type of event to listen for.
 * @param block The block that should be called with initial data and updates as a FDataSnapshot, as well as the previous child's key.
 * @param cancelBlock The block that will be called if you don't have permission to access this data
 */
- (void) observeSingleEventOfType:(FEventType)eventType andPreviousSiblingKeyWithBlock:(void (^)(FDataSnapshot* snapshot, NSString* prevKey))block withCancelBlock:(void (^)(NSError* error))cancelBlock;

/** @name Detaching observers */

/**
 * Detach a block previously attached with observeEventType:withBlock:.
 *
 * @param handle The handle returned by the call to observeEventType:withBlock: which we are trying to remove.
 */
- (void) removeObserverWithHandle:(FirebaseHandle)handle;

/**
 * By calling `keepSynced:YES` on a location, the data for that location will automatically be downloaded and
 * kept in sync, even when no listeners are attached for that location. Additionally, while a location is kept
 * synced, it will not be evicted from the persistent disk cache.
 *
 * @param keepSynced Pass YES to keep this location synchronized, pass NO to stop synchronization.
 */
- (void) keepSynced:(BOOL)keepSynced;


/**
 * Detach all blocks previously attached to this Firebase location with observeEventType:withBlock:
 */
- (void) removeAllObservers;

/** @name Querying and limiting */


/**
 * This method is deprecated in favor of using queryStartingAtValue:. This can be used with queryOrderedByPriority
 * to query by priority.
 *
 * queryStartingAtPriority: is used to generate a reference to a limited view of the data at this location.
 * The FQuery instance returned by queryStartingAtPriority: will respond to events at nodes with a priority
 * greater than or equal to startPriority
 *
 * @param startPriority The lower bound, inclusive, for the priority of data visible to the returned FQuery
 * @return An FQuery instance, limited to data with priority greater than or equal to startPriority
 */
- (FQuery *) queryStartingAtPriority:(id)startPriority __attribute__((deprecated("Use [[FQuery queryOrderedByPriority] queryStartingAtValue:] instead")));


/**
 * This method is deprecated in favor of using queryStartingAtValue:childKey:. This can be used with queryOrderedByPriority
 * to query by priority.
 *
 * queryStartingAtPriority:andChildName: is used to generate a reference to a limited view of the data at this location.
 * The FQuery instance returned by queryStartingAtPriority:andChildName will respond to events at nodes with a priority
 * greater than startPriority, or equal to startPriority and with a name greater than or equal to childName
 *
 * @param startPriority The lower bound, inclusive, for the priority of data visible to the returned FQuery
 * @param childName The lower bound, inclusive, for the name of nodes with priority equal to startPriority
 * @return An FQuery instance, limited to data with priority greater than or equal to startPriority
 */
- (FQuery *) queryStartingAtPriority:(id)startPriority andChildName:(NSString *)childName __attribute__((deprecated("Use [[FQuery queryOrderedByPriority] queryStartingAtValue:childKey:] instead")));

/**
 * This method is deprecated in favor of using queryEndingAtValue:. This can be used with queryOrderedByPriority
 * to query by priority.
 *
 * queryEndingAtPriority: is used to generate a reference to a limited view of the data at this location.
 * The FQuery instance returned by queryEndingAtPriority: will respond to events at nodes with a priority
 * less than or equal to startPriority and with a name greater than or equal to childName
 *
 * @param endPriority The upper bound, inclusive, for the priority of data visible to the returned FQuery
 * @return An FQuery instance, limited to data with priority less than or equal to endPriority
 */
- (FQuery *) queryEndingAtPriority:(id)endPriority __attribute__((deprecated("Use [[FQuery queryOrderedByPriority] queryEndingAtValue:] instead")));


/**
 * This method is deprecated in favor of using queryEndingAtValue:childKey:. This can be used with queryOrderedByPriority
 * to query by priority.
 *
 * queryEndingAtPriority:andChildName: is used to generate a reference to a limited view of the data at this location.
 * The FQuery instance returned by queryEndingAtPriority:andChildName will respond to events at nodes with a priority
 * less than endPriority, or equal to endPriority and with a name less than or equal to childName
 *
 * @param endPriority The upper bound, inclusive, for the priority of data visible to the returned FQuery
 * @param childName The upper bound, inclusive, for the name of nodes with priority equal to endPriority
 * @return An FQuery instance, limited to data with priority less than endPriority or equal to endPriority and with a name less than or equal to childName
 */
- (FQuery *) queryEndingAtPriority:(id)endPriority andChildName:(NSString *)childName __attribute__((deprecated("Use [[FQuery queryOrderedByPriority] queryEndingAtValue:childKey:] instead")));


/**
 * This method is deprecated in favor of using queryEqualToValue:. This can be used with queryOrderedByPriority
 * to query by priority.
 *
 * queryEqualToPriority: is used to generate a reference to a limited view of the data at this location.
 * The FQuery instance returned by queryEqualToPriority: will respond to events at nodes with a priority equal to
 * supplied argument.
 *
 * @param priority The priority that the data returned by this FQuery will have
 * @return An Fquery instance, limited to data with the supplied priority.
 */
- (FQuery *) queryEqualToPriority:(id)priority __attribute__((deprecated("Use [[FQuery queryOrderedByPriority] queryEqualToValue:] instead")));


/**
 * This method is deprecated in favor of using queryEqualAtValue:childKey:. This can be used with queryOrderedByPriority
 * to query by priority.
 *
 * queryEqualToPriority:andChildName: is used to generate a reference to a limited view of the data at this location.
 * The FQuery instance returned by queryEqualToPriority:andChildName will respond to events at nodes with a priority
 * equal to the supplied argument with a name equal to childName. There will be at most one node that matches because
 * child names are unique.
 *
 * @param priority The priority that the data returned by this FQuery will have
 * @param childName The name of nodes with the right priority
 * @return An FQuery instance, limited to data with the supplied priority and the name.
 */
- (FQuery *) queryEqualToPriority:(id)priority andChildName:(NSString *)childName __attribute__((deprecated("Use [[FQuery queryOrderedByPriority] queryEqualToValue:childKey:] instead")));

/**
 * This method is deprecated in favor of using queryLimitedToFirst:limit or queryLimitedToLast:limit instead.
 *
 * queryLimitedToNumberOfChildren: is used to generate a reference to a limited view of the data at this location.
 * The FQuery instance returned by queryLimitedToNumberOfChildren: will respond to events from at most limit child nodes.
 *
 * @param limit The upper bound, inclusive, for the number of child nodes to receive events for
 * @return An FQuery instance, limited to at most limit child nodes.
 */
- (FQuery *) queryLimitedToNumberOfChildren:(NSUInteger)limit __attribute__((deprecated("Use [FQuery queryLimitedToFirst:limit] or [FQuery queryLimitedToLast:limit] instead")));


/**
 * queryLimitedToFirst: is used to generate a reference to a limited view of the data at this location.
 * The FQuery instance returned by queryLimitedToFirst: will respond to at most the first limit child nodes.
 *
 * @param limit The upper bound, inclusive, for the number of child nodes to receive events for
 * @return An FQuery instance, limited to at most limit child nodes.
 */
- (FQuery *) queryLimitedToFirst:(NSUInteger)limit;


/**
 * queryLimitedToLast: is used to generate a reference to a limited view of the data at this location.
 * The FQuery instance returned by queryLimitedToLast: will respond to at most the last limit child nodes.
 *
 * @param limit The upper bound, inclusive, for the number of child nodes to receive events for
 * @return An FQuery instance, limited to at most limit child nodes.
 */
- (FQuery *) queryLimitedToLast:(NSUInteger)limit;

/**
 * queryOrderBy: is used to generate a reference to a view of the data that's been sorted by the values of
 * a particular child key. This method is intended to be used in combination with queryStartingAtValue:,
 * queryEndingAtValue:, or queryEqualToValue:.
 *
 * @param key The child key to use in ordering data visible to the returned FQuery
 * @return An FQuery instance, ordered by the values of the specified child key.
 */
- (FQuery *) queryOrderedByChild:(NSString *)key;

/**
 * queryOrderedByKey: is used to generate a reference to a view of the data that's been sorted by child key.
 * This method is intended to be used in combination with queryStartingAtValue:, queryEndingAtValue:,
 * or queryEqualToValue:.
 *
 * @return An FQuery instance, ordered by child keys.
 */
- (FQuery *) queryOrderedByKey;

/**
 * queryOrderedByPriority: is used to generate a reference to a view of the data that's been sorted by child
 * priority. This method is intended to be used in combination with queryStartingAtValue:, queryEndingAtValue:,
 * or queryEqualToValue:.
 *
 * @return An FQuery instance, ordered by child priorities.
 */
- (FQuery *) queryOrderedByPriority;

/**
 * queryStartingAtValue: is used to generate a reference to a limited view of the data at this location.
 * The FQuery instance returned by queryStartingAtValue: will respond to events at nodes with a value
 * greater than or equal to startValue.
 *
 * @param startValue The lower bound, inclusive, for the value of data visible to the returned FQuery
 * @return An FQuery instance, limited to data with value greater than or equal to startValue
 */
- (FQuery *) queryStartingAtValue:(id)startValue;

/**
 * queryStartingAtValue:childKey: is used to generate a reference to a limited view of the data at this location.
 * The FQuery instance returned by queryStartingAtValue:childKey will respond to events at nodes with a value
 * greater than startValue, or equal to startValue and with a key greater than or equal to childKey.
 *
 * @param startValue The lower bound, inclusive, for the value of data visible to the returned FQuery
 * @param childKey The lower bound, inclusive, for the key of nodes with value equal to startValue
 * @return An FQuery instance, limited to data with value greater than or equal to startValue
 */
- (FQuery *) queryStartingAtValue:(id)startValue childKey:(NSString *)childKey;

/**
 * queryEndingAtValue: is used to generate a reference to a limited view of the data at this location.
 * The FQuery instance returned by queryEndingAtValue: will respond to events at nodes with a value
 * less than or equal to endValue.
 *
 * @param endValue The upper bound, inclusive, for the value of data visible to the returned FQuery
 * @return An FQuery instance, limited to data with value less than or equal to endValue
 */
- (FQuery *) queryEndingAtValue:(id)endValue;

/**
 * queryEndingAtValue:childKey: is used to generate a reference to a limited view of the data at this location.
 * The FQuery instance returned by queryEndingAtValue:childKey will respond to events at nodes with a value
 * less than endValue, or equal to endValue and with a key less than or equal to childKey.
 *
 * @param endValue The upper bound, inclusive, for the value of data visible to the returned FQuery
 * @param childKey The upper bound, inclusive, for the key of nodes with value equal to endValue
 * @return An FQuery instance, limited to data with value less than or equal to endValue
 */
- (FQuery *) queryEndingAtValue:(id)endValue childKey:(NSString *)childKey;

/**
 * queryEqualToValue: is used to generate a reference to a limited view of the data at this location.
 * The FQuery instance returned by queryEqualToValue: will respond to events at nodes with a value equal
 * to the supplied argument.
 *
 * @param value The value that the data returned by this FQuery will have
 * @return An Fquery instance, limited to data with the supplied value.
 */
- (FQuery *) queryEqualToValue:(id)value;

/**
 * queryEqualToValue:childKey: is used to generate a reference to a limited view of the data at this location.
 * The FQuery instance returned by queryEqualToValue:childKey will respond to events at nodes with a value
 * equal to the supplied argument with a name equal to childKey. There will be at most one node that matches because
 * child keys are unique.
 *
 * @param value The value that the data returned by this FQuery will have
 * @param childKey The name of nodes with the right value
 * @return An FQuery instance, limited to data with the supplied value and the key.
 */
- (FQuery *) queryEqualToValue:(id)value childKey:(NSString *)childKey;

/** @name Managing presence */

/**
 * Ensure the data at this location is set to the specified value when
 * the client is disconnected (due to closing the browser, navigating
 * to a new page, or network issues).
 *
 * onDisconnectSetValue: is especially useful for implementing "presence" systems,
 * where a value should be changed or cleared when a user disconnects
 * so that he appears "offline" to other users.
 *
 * @param value The value to be set after the connection is lost.
 */
- (void) onDisconnectSetValue:(id)value;


/**
 * Ensure the data at this location is set to the specified value when
 * the client is disconnected (due to closing the browser, navigating
 * to a new page, or network issues).
 *
 * The completion block will be triggered when the operation has been successfully queued up on the Firebase servers
 *
 * @param value The value to be set after the connection is lost.
 * @param block Block to be triggered when the operation has been queued up on the Firebase servers
 */
- (void) onDisconnectSetValue:(id)value withCompletionBlock:(void (^)(NSError* error, Firebase* ref))block;


/**
 * Ensure the data at this location is set to the specified value and priority when
 * the client is disconnected (due to closing the browser, navigating
 * to a new page, or network issues).
 *
 * @param value The value to be set after the connection is lost.
 * @param priority The priority to be set after the connection is lost.
 */
- (void) onDisconnectSetValue:(id)value andPriority:(id)priority;


/**
 * Ensure the data at this location is set to the specified value and priority when
 * the client is disconnected (due to closing the browser, navigating
 * to a new page, or network issues).
 *
 * The completion block will be triggered when the operation has been successfully queued up on the Firebase servers
 *
 * @param value The value to be set after the connection is lost.
 * @param priority The priority to be set after the connection is lost.
 * @param block Block to be triggered when the operation has been queued up on the Firebase servers
 */
- (void) onDisconnectSetValue:(id)value andPriority:(id)priority withCompletionBlock:(void (^)(NSError* error, Firebase* ref))block;


/**
 * Ensure the data at this location is removed when
 * the client is disconnected (due to closing the app, navigating
 * to a new page, or network issues).
 *
 * onDisconnectRemoveValue is especially useful for implementing "presence" systems.
 */
- (void) onDisconnectRemoveValue;


/**
 * Ensure the data at this location is removed when
 * the client is disconnected (due to closing the app, navigating
 * to a new page, or network issues).
 *
 * onDisconnectRemoveValueWithCompletionBlock: is especially useful for implementing "presence" systems.
 *
 * @param block Block to be triggered when the operation has been queued up on the Firebase servers
 */
- (void) onDisconnectRemoveValueWithCompletionBlock:(void (^)(NSError* error, Firebase* ref))block;



/**
 * Ensure the data has the specified child values updated when
 * the client is disconnected (due to closing the browser, navigating
 * to a new page, or network issues).
 *
 *
 * @param values A dictionary of child node keys and the values to set them to after the connection is lost.
 */
- (void) onDisconnectUpdateChildValues:(NSDictionary *)values;


/**
 * Ensure the data has the specified child values updated when
 * the client is disconnected (due to closing the browser, navigating
 * to a new page, or network issues).
 *
 *
 * @param values A dictionary of child node keys and the values to set them to after the connection is lost.
 * @param block A block that will be called once the operation has been queued up on the Firebase servers
 */
- (void) onDisconnectUpdateChildValues:(NSDictionary *)values withCompletionBlock:(void (^)(NSError* error, Firebase* ref))block;


/**
 * Cancel any operations that are set to run on disconnect. If you previously called onDisconnectSetValue:,
 * onDisconnectRemoveValue:, or onDisconnectUpdateChildValues:, and no longer want the values updated when the 
 * connection is lost, call cancelDisconnectOperations:
 */
- (void) cancelDisconnectOperations;


/**
 * Cancel any operations that are set to run on disconnect. If you previously called onDisconnectSetValue:,
 * onDisconnectRemoveValue:, or onDisconnectUpdateChildValues:, and no longer want the values updated when the
 * connection is lost, call cancelDisconnectOperations:
 *
 * @param block A block that will be triggered once the Firebase servers have acknowledged the cancel request.
 */
- (void) cancelDisconnectOperationsWithCompletionBlock:(void (^)(NSError* error, Firebase* ref))block;


/** @name Reading and observing authentication data */

/**
* Get the authentication data of the current user.
*
* @return Authentication data of the current user.
*/
@property (nonatomic, strong, readonly) FAuthData *authData;


/**
* Observer block will be triggered whenever a user gets authenticated or logged out.
*
* Authentication data is persisted across app restarts. If your have an old authentication, Firebase will attempt to
* resume your old session. This approach does not wait for a server roundtrip. Rather, it inspects the
* contents of the persisted JWT and assumes that the Firebase secret used to generate the token has not been revoked.
*
* In the event that the Firebase secret used to generate the token has been revoked, observers will likely see one
* flicker / rapid flip-flop of authentication state once the server rejects the token.
*
* Use removeAuthEventObserverWithHandle: to stop receiving updates.
*
* @param block The block that should be called with initial authentication data and future updates
* @return A handle used to unregister this block later with removeAuthEventObserverWithHandle:
*/
- (FirebaseHandle) observeAuthEventWithBlock:(void (^)(FAuthData *authData))block;


/**
* Detach a block previously attached with observeAuthEventWithBlock:.
*
* @param handle The handle returned by the call to observeAuthEventWithBlock: which we are trying to remove.
*/
- (void) removeAuthEventObserverWithHandle:(FirebaseHandle)handle;

/** @name User creation and modification */

/**
* Used to create a new user account with the given email and password combo. The results will be passed to the given block.
* Note that this method will not log the new user in.
*
* @param email The email for the account to be created
* @param password The password for the account to be created
* @param block The block to be called with the results of the operation
*/
- (void) createUser:(NSString *)email password:(NSString *)password withCompletionBlock:(void (^)(NSError *error))block;

/**
 * Used to create a new user account with the given email and password combo. The results will be passed
 * to the given block. Note that this method will not log the new user in. On success, invokes the result
 * block with an dictionary of user data, including the user id.
 *
 * @param email The email for the account to be created
 * @param password The password for the account to be created
 * @param block The block to be called with the results of the operation
 */
- (void) createUser:(NSString *)email password:(NSString *)password withValueCompletionBlock:(void (^)(NSError *error, NSDictionary *result))block;

/**
* Remove a user account with the given email and password.
*
* @param email The email of the account to be removed
* @param password The password for the account to be removed
* @param block A block to receive the results of the operation
*/
- (void) removeUser:(NSString *)email password:(NSString *)password withCompletionBlock:(void (^)(NSError *error))block;


/**
* Attempts to change the password for the account with the given credentials to the new password given. Results are reported to the supplied block.
*
* @param email The email for the account to be changed
* @param oldPassword The old password for the account to be changed
* @param newPassword The desired newPassword for the account
* @param block A block to receive the results of the operation
*/
- (void) changePasswordForUser:(NSString *)email fromOld:(NSString *)oldPassword toNew:(NSString *)newPassword withCompletionBlock:(void (^)(NSError *error))block;


/**
 * Attempts to change the email for the account with the given credentials to the new email given. Results are reported to the supplied block.
 *
 * @param email The email for the account to be changed
 * @param password The password for the account to be changed
 * @param newEmail The desired newEmail for the account
 * @param block A block to receive the results of the operation
 */
- (void) changeEmailForUser:(NSString *)email password:(NSString *)password toNewEmail:(NSString *)newEmail withCompletionBlock:(void (^)(NSError *error))block;


/**
* Send a password reset email to the owner of the account with the given email. Results are reported to the supplied block.
*
* @param email The email of the account to be removed
* @param block A block to receive the results of the operation
*/
- (void) resetPasswordForUser:(NSString *)email withCompletionBlock:(void (^)(NSError* error))block;

/** @name Authenticating */

/**
* Attempts to log the user in anonymously. The block will receive the results of the attempt.
*
* @param block A block to receive the results of the authentication attempt.
*/
- (void) authAnonymouslyWithCompletionBlock:(void (^)(NSError *error, FAuthData *authData))block;

/**
* Attempts to authenticate to Firebase with the given credentials. The block will receive the results of the attempt.
*
* @param email The email of the account
* @param password The password for the account
* @param block A block to receive the results of the authentication attempt
*/
- (void) authUser:(NSString *)email password:(NSString *)password withCompletionBlock:(void (^)(NSError *error, FAuthData *authData))block;

/**
* Authenticate access to this Firebase using the provided credentials.
*
* The completion block will be called with the results of the authenticated attempt. Unlike
* authWithCredential:withCompletionBlock:withCancelBlock:, no block will be called when the credentials become invalid.
*
* Instead, please use observeAuthEventWithBlock: to observe if a user gets logged out.
*
* @param token The Firebase authentication JWT generated by a secure code on a remote server.
* @param block This block will be called with the results of the authentication attempt
*/
- (void) authWithCustomToken:(NSString *)token withCompletionBlock:(void (^)(NSError *error, FAuthData *authData))block;

/**
* Authenticate to Firebase with an OAuth token from a provider.
*
* This method works with current OAuth 2.0 providers such as Facebook, Google+, and Github.
*
* For other providers that Firebase supports which require additional parameters for login, such as Twitter, please use authWithOAuthProvider:parameters:withCompletionBlock:.
*
* @param provider The provider, all lower case with no spaces.
* @param oauthToken The OAuth Token to authenticate with the provider
* @param block A block to receive the results of the authentication attempt
*/
- (void) authWithOAuthProvider:(NSString *)provider token:(NSString *)oauthToken withCompletionBlock:(void (^) (NSError *error, FAuthData *authData))block;

/**
* Authenticate to Firebase with an OAuth token from a provider.
*
* This method is for OAuth providers that require extra parameters when authentication with the server, such as Twitter.
* The OAuth token should be included as a parameter.
*
* @param provider The provider, all lowercase with no spaces.
* @param parameters The parameters necessary to authenticate with the provider
* @param block A block to receive the results of the authentication attempt
*/
- (void) authWithOAuthProvider:(NSString *)provider parameters:(NSDictionary *)parameters withCompletionBlock:(void (^) (NSError *error, FAuthData *authData))block;

/**
* Make a reverse OAuth Request to a provider.
*
* This method is for OAuth providers that require a reverse request be made first. The json output of this block
*
* @param provider The provider, all lowercase with no spaces.
* @param block The block to receive the results of the reverse OAuth request.
*/
- (void) makeReverseOAuthRequestTo:(NSString *)provider withCompletionBlock:(void (^)(NSError *error, NSDictionary *json))block;

/**
* Removes any credentials associated with this Firebase.
*/
- (void) unauth;

/**
* This method is deprecated. Use authWithCustomToken:withCompletionBlock: instead.
*
* Authenticate access to this Firebase using the provided credentials. The completion block will be called with
* the results of the authenticated attempt, and the cancelBlock will be called if the credentials become invalid
* at some point after authentication has succeeded.
*
* @param credential The Firebase authentication JWT generated by a secure code on a remote server.
* @param block This block will be called with the results of the authentication attempt
* @param cancelBlock This block will be called if at any time in the future the credentials become invalid
*/
- (void) authWithCredential:(NSString *)credential withCompletionBlock:(void (^) (NSError* error, id data))block withCancelBlock:(void (^)(NSError* error))cancelBlock __attribute__((deprecated("Use authWithCustomToken:withCompletionblock: instead")));

/**
* This method is deprecated. Use unauth: instead.
* 
* Removes any credentials associated with this Firebase. The callback block will be triggered after this operation
* has been acknowledged by the Firebase servers.
*
* @param block This block will be called once the unauth has completed.
*/
- (void) unauthWithCompletionBlock:(void (^)(NSError* error))block __attribute__((deprecated("Use unauth: instead")));


/** @name Manual Connection Management */

/**
 * Manually disconnect the Firebase client from the server and disable automatic reconnection.
 *
 * The Firebase client automatically maintains a persistent connection to the Firebase server, 
 * which will remain active indefinitely and reconnect when disconnected. However, the goOffline( ) 
 * and goOnline( ) methods may be used to manually control the client connection in cases where 
 * a persistent connection is undesirable.
 * 
 * While offline, the Firebase client will no longer receive data updates from the server. However, 
 * all Firebase operations performed locally will continue to immediately fire events, allowing 
 * your application to continue behaving normally. Additionally, each operation performed locally 
 * will automatically be queued and retried upon reconnection to the Firebase server.
 * 
 * To reconnect to the Firebase server and begin receiving remote events, see goOnline( ). 
 * Once the connection is reestablished, the Firebase client will transmit the appropriate data 
 * and fire the appropriate events so that your client "catches up" automatically.
 * 
 * Note: Invoking this method will impact all Firebase connections. 
 */
+ (void) goOffline;

/**
 * Manually reestablish a connection to the Firebase server and enable automatic reconnection.
 *
 * The Firebase client automatically maintains a persistent connection to the Firebase server, 
 * which will remain active indefinitely and reconnect when disconnected. However, the goOffline( ) 
 * and goOnline( ) methods may be used to manually control the client connection in cases where 
 * a persistent connection is undesirable.
 * 
 * This method should be used after invoking goOffline( ) to disable the active connection. 
 * Once reconnected, the Firebase client will automatically transmit the proper data and fire 
 * the appropriate events so that your client "catches up" automatically.
 * 
 * To disconnect from the Firebase server, see goOffline( ).
 * 
 * Note: Invoking this method will impact all Firebase connections.
 */
+ (void) goOnline;


/** @name Transactions */

/**
 * Performs an optimistic-concurrency transactional update to the data at this location. Your block will be called with an FMutableData
 * instance that contains the current data at this location. Your block should update this data to the value you
 * wish to write to this location, and then return an instance of FTransactionResult with the new data.
 *
 * If, when the operation reaches the server, it turns out that this client had stale data, your block will be run
 * again with the latest data from the server.
 *
 * When your block is run, you may decide to abort the transaction by return [FTransactionResult abort].
 *
 * @param block This block receives the current data at this location and must return an instance of FTransactionResult
 */
- (void) runTransactionBlock:(FTransactionResult* (^) (FMutableData* currentData))block;


/**
 * Performs an optimistic-concurrency transactional update to the data at this location. Your block will be called with an FMutableData
 * instance that contains the current data at this location. Your block should update this data to the value you
 * wish to write to this location, and then return an instance of FTransactionResult with the new data.
 *
 * If, when the operation reaches the server, it turns out that this client had stale data, your block will be run
 * again with the latest data from the server.
 *
 * When your block is run, you may decide to abort the transaction by return [FTransactionResult abort].
 *
 * @param block This block receives the current data at this location and must return an instance of FTransactionResult
 * @param completionBlock This block will be triggered once the transaction is complete, whether it was successful or not. It will indicate if there was an error, whether or not the data was committed, and what the current value of the data at this location is.
 */
- (void) runTransactionBlock:(FTransactionResult* (^) (FMutableData* currentData))block andCompletionBlock:(void (^) (NSError* error, BOOL committed, FDataSnapshot* snapshot))completionBlock;



/**
 * Performs an optimistic-concurrency transactional update to the data at this location. Your block will be called with an FMutableData
 * instance that contains the current data at this location. Your block should update this data to the value you
 * wish to write to this location, and then return an instance of FTransactionResult with the new data.
 *
 * If, when the operation reaches the server, it turns out that this client had stale data, your block will be run
 * again with the latest data from the server.
 *
 * When your block is run, you may decide to abort the transaction by return [FTransactionResult abort].
 *
 * Since your block may be run multiple times, this client could see several immediate states that don't exist on the server. You can suppress those immediate states until the server confirms the final state of the transaction.
 *
 * @param block This block receives the current data at this location and must return an instance of FTransactionResult
 * @param completionBlock This block will be triggered once the transaction is complete, whether it was successful or not. It will indicate if there was an error, whether or not the data was committed, and what the current value of the data at this location is.
 * @param localEvents Set this to NO to suppress events raised for intermediate states, and only get events based on the final state of the transaction.
 */
- (void) runTransactionBlock:(FTransactionResult* (^) (FMutableData* currentData))block andCompletionBlock:(void (^) (NSError* error, BOOL committed, FDataSnapshot* snapshot))completionBlock withLocalEvents:(BOOL)localEvents;


/** @name Retrieving String Representation */

/**
 * Gets the absolute URL of this Firebase location. 
 *
 * @return The absolute URL of the referenced Firebase location.
 */
- (NSString *) description;

/** @name Properties */

/**
 * Get a Firebase reference for the parent location.
 * If this instance refers to the root of your Firebase, it has no parent,
 * and therefore parent( ) will return null.
 *
 * @return A Firebase reference for the parent location.
 */
@property (strong, readonly, nonatomic) Firebase* parent;


/**
 * Get a Firebase reference for the root location
 *
 * @return A new Firebase reference to root location.
 */
@property (strong, readonly, nonatomic) Firebase* root;


/**
 * Gets last token in a Firebase location (e.g. 'fred' in https://SampleChat.firebaseIO-demo.com/users/fred)
 *
 * @return The key of the location this reference points to.
 */
@property (strong, readonly, nonatomic) NSString* key;

/**
 * Gets the FirebaseApp instance associated with this reference.
 *
 * @return The FirebaseApp object for this reference.
 */
@property (strong, readonly, nonatomic) FirebaseApp *app;


/** @name Global configuration and settings */

/** Set the default dispatch queue for event blocks.
 *
 * @param queue The queue to set as the default for running blocks for all Firebase event types.
 * @deprecated This method is deprecated
 * @note Please use [Firebase defaultConfig].callbackQueue instead
*/
+ (void) setDispatchQueue:(dispatch_queue_t)queue __attribute__((deprecated));

/** Retrieve the Firebase SDK version. */
+ (NSString *) sdkVersion;

+ (void) setLoggingEnabled:(BOOL)enabled;

/**
 * Returns the default config object, used for configuring Firebase client behavior.
 *
 * This can be modified until you create your first `Firebase` instance.
 */
+ (FConfig *)defaultConfig;

/**
 * @deprecated This method is deprecated
 * @note Please enable persistence by setting [Firebase defaultConfig].persistenceEnabled = YES instead.
 * @param option Option to set.
 * @param value Value to set.
 */
+ (void) setOption:(NSString*)option to:(id)value __attribute__((deprecated));
@end
